# Cleaned and simplified version of the BTFR plot used in the AGES Leo paper (figure 7). 
# User supplies two files :
# 1) The AGES data from VC1
# 2) McGaugh's data (resolved rotation curves)
# The McGaugh file contains the data already corrected and is just used for plotting.
# The AGES data file contains parameters taken directly from the AGES measurements plus a few others :
# - Apparent g, i magnitudes; extinctions; b/a from SDSS; CMB velocity (see Leo paper appendix A.1).
# Produces a single plot with AGES shown as black filled circles and McGaugh's as open black circles.
# Also shows McGaguh's relation with +/- 1, 2 sigma.


import os
import numpy
import matplotlib
matplotlib.use('agg')
from matplotlib import pyplot
import matplotlib.pyplot as plt
import math

# Change to directory where script is located
abspath = os.path.abspath(__file__)
dname = os.path.dirname(abspath)
os.chdir(dname)


# USER-DEFINED PARAMETERS
# INPUT FILES
virgofilename       = dname+'/InputTables/VC1GalaxiesClean.txt'		# AGES Virgo galaxies
mggalaxiesfilename  = dname+'/InputTables/McGaughGas-RichBTFR.txt'	# McGaugh galaxies

# ASTROPHYSICAL VALUES
# Correction for helium according to McGaugh (most other people use 1.36)
xHeMg = 1.33

# Solar absolute magnitudes
gsol = 5.36
isol = 4.58		# For consistency with Taylor et al. 2011. Other sources say 4.48.

# Inclination angle limit - below this, velocity correction is too susceptible to measurement errors
# (this is conservative, can be reduced to 30 degrees)
# Converting i to radians here means we never have to do the conversion again later on.
imin = math.radians(45.0)
# HI deficiency upper limit - below this, the line width may be affected by gas loss
# (again conservative, strong deviations not usually seen below deficiency = 1.0)
HIdefmax = 0.6


# PLOTTING SETUP
# LaTeX like fonts in the figure
plt.rc('font', family='serif')

titlessize = 15
numbersize = 12

from matplotlib.ticker import AutoMinorLocator
XminorLocator = AutoMinorLocator()
YminorLocator = AutoMinorLocator()

# Disable interactive plotting
pyplot.ioff()

# Set axes limits
vmin = 0.8
vmax = 2.6
bmin = 6.4
bmax = 11.2


# MAIN CODE
# Correct for instrumental smoothing - generates a correction deltaS which should be linearly subtracted from
# the initial velocity width
def deltaS(channelwidth, peakflux, rms):

	deltaV = channelwidth	# Channel width, km/s
	SNforCorrection = math.log10( (peakflux - rms) / rms )
	
	if 'nan' in str(SNforCorrection):
		Lmbda = math.nan
	if SNforCorrection < 0.6:
		Lmbda = 0.037*deltaV - 0.18
	if SNforCorrection >= 0.6 and SNforCorrection < 1.1:
		Lmbda = (0.0527*deltaV - 0.732) + ((-0.027*deltaV + 0.92)*SNforCorrection)
	if SNforCorrection >= 1.1:
		Lmbda = 0.023*deltaV + 0.28
	if 'nan' not in str(Lmbda):
		deltaS = 2.0*deltaV*Lmbda		# Correction for instrumental effects (smoothing)
		return deltaS
	else:
		return math.nan
	

# Correction for redshift
def RedShift(velocity):
	z = (velocity*1000.0) / 299792458.0
	RedShift = (1 + z)
	
	return RedShift


# Calculate the total integrated S/N according to ALFALFA (Saintonge 2007)
# NOT USED. But might be a useful additional/alternative cut on the data
def aasn(totflux, w50, vres, rms):
	
	if w50 < 400.0:
		wsmo = w50 / (2.0*vres)
	if w50 >= 400.0:
		wsmo = 400.0 / (2.0*vres)
			
	intsn = (1000.0*totflux / w50) * (math.sqrt(wsmo) / rms)
	
	return intsn


# READ IN THE DATA
# The main AGES data file
VirgoFile = open(virgofilename,'r')

# Arrays to hold the final corrected baryonic mass and rotation velocity. Somewhat deprected naming
# syntax : data is "mixed" in the sense that it uses a combination of pure AGES and other corrected
# factors (this made more sense when the code was experimenting with using different measurements !)
mixedagesaa_mcg_b = []		# Baryonic mass, using McGaugh convetion for stellar mass
mixedagesaa_v20_mcg_b = []	# Circular velocity, using McGaugh corrections

# Produce an OUTPUT file to hold the final data. Not used in the plot, but can be very useful when
# individual corrected measurements are of interest
ComparisonFile = open('DataCheckFile_new.txt','w')
ComparisonFile.write('#ID V20	Mgas	Mstar	Mbar Distance'+'\n')

# READ IN THE DATA AND APPLY THE CORRECTIONS
for line in VirgoFile:
	if '#' in line:
		continue
	else:
		ID, RA, Dec, Velocity, W50_AGES, W20_AGES, Distance, MHI_corrected, m_g_AGES, m_i_AGES, HIdef_AGES, Ag_AGES, Ai_AGES, SDSS_bovera, PFlux_AGES, TotFlux_AGES, rms_AGES, VelCMB = line.split()

		# Set unknown deficiencies to be ludicrously high so they'll be ignored
		if HIdef_AGES == '""':
			HIdef_AGES = 10.0


		# Use the inclination from the SDSS
		AA_i = math.acos(float(SDSS_bovera))
						
		# Only do anything for reliable inclination angles and low deficiencies
		if AA_i >= imin and float(HIdef_AGES) < HIdefmax and float(PFlux_AGES)/(5.0*float(rms_AGES)) > 2.0:
			
			# For objects outside Virgo, use the ALFALFA distance formulation. This is only very marginally different to the
			# AGES convention, but this ensures everything should be identical
			# NOTE : The special case of distances < 32.0 Mpc really only applies to the Virgo Cluster, where distances are
			# assigned individually. For other fields, generally the distances can all be set as velcmb / 70.0.
			if float(Distance) > 32.0:
				AA_Distance = float(VelCMB) / 70.0
			if float(Distance) <= 32.0:
				AA_Distance = float(Distance)
				
			W50_AGES = float(W50_AGES)
			W20_AGES = float(W20_AGES)

		
			# Stellar masses. First correct for galactic extinction
			m_ge = float(m_g_AGES) - float(Ag_AGES)
			m_ie = float(m_i_AGES) - float(Ai_AGES)
		
			# Convert to absolute magnitudes
			M_ge = -(5.0*math.log10(AA_Distance*1000000.0) -5.0 - m_ge)
			M_ie = -(5.0*math.log10(AA_Distance*1000000.0) -5.0 - m_ie)
		
			# Correct for internal extinction, following Durbala+2020. Multi-stage process !
			gam_g = 0.0
			if M_ge < -17.0:
				gam_g = -0.35*M_ge - 5.95
			
			gam_i = 0.0
			if M_ie < -17.0:
				gam_i = -0.15*M_ie - 2.55
			
			A_gamg = gam_g*math.log10(math.cos(AA_i))
			A_gami = gam_i*math.log10(math.cos(AA_i))
		
			# Now we apply these additional attenuation factors to the absolute magntidues
			M_gee = M_ge + A_gamg
			M_iee = M_ie + A_gami
						
			# With our corrected magnitudes we can calculate luminosities, colours, and stellar masses
			L_gee = 10.0**( (M_gee-gsol)/-2.5 )
			L_iee = 10.0**( (M_iee-isol)/-2.5 )
		
			col_giee = M_gee - M_iee
		
			Mstar_AGES = 10.0**( -0.68 + 0.7*col_giee + math.log10(L_iee))
				
				
			# Corrections to the gas properties
			# 1) Mass correction for helium. First correct the HI mass itself.
			# Another correction to use the same conventions as ALFALFA. Again, this is a Virgo-specific
			# thing in which everything below 32.0 Mpc is effectively set manually because distances so
			# we use the table value. For other fields the mass should use the first correction.
			if float(Distance) > 32.0: 
				MHI_AGES_New = 2.36E5*AA_Distance*AA_Distance*float(TotFlux_AGES)	# <- Everything in Leo should use this
			if float(Distance) <= 32.0:
				MHI_AGES_New = float(MHI_corrected)
			
			# Correction for helium	
			Mgas_AGES = float(MHI_AGES_New)*xHeMg
			
			# 2) Velocity width corrections
			# a) Instrumental smoothing							
			W20_AGES = W20_AGES - deltaS(10.6, float(PFlux_AGES), float(rms_AGES))
			W50_AGES = W50_AGES - deltaS(10.6, float(PFlux_AGES), float(rms_AGES))
			
			# b) Redshift correction
			W20_AGES = W20_AGES / RedShift(float(Velocity)) 
			W50_AGES = W50_AGES / RedShift(float(Velocity)) 
			
			# c)i) Rotation is half the line width
			V20_AGES = W20_AGES / (2.0*math.sin(AA_i))
			V50_AGES = W50_AGES / (2.0*math.sin(AA_i))
			
			# c)ii) McGaugh extra correction for line width measuring Vmax rather than Vflat:
			if Mstar_AGES > Mgas_AGES:
				V20_AGES = W20_AGES / (2.0*1.1*math.sin(AA_i))
				V50_AGES = W50_AGES / (2.0*1.1*math.sin(AA_i))
					
			# We can already get the baryonic mass, but this is preliminary - we have to do some 
			# further jiggery-pokery...
			Mbar_AGES = Mstar_AGES + Mgas_AGES
			
			# But, Durbala also says how to correct the Taylor stellar masses to the McGaugh version, so do this for AGES as well !
			# Have to do this in stages, unfortunately... first convert to GSWLC :
			logMstarAGES_C = 1.052*math.log10(Mstar_AGES) - 0.369
			logMstarAGES_B = (logMstarAGES_C + 0.9755) / 1.084
			Mstar_AGES_B = 10.0**logMstarAGES_B
			
			# As above, this means velocity corrections as well. We don't have to change the gas mass, as this is dependent only on
			# the HI mass, but line width corrections depend on the gas to stellar mass ratio 
			# a) Default, gas-dominated galaxies
			V20_AGES_B = W20_AGES / (2.0*math.sin(AA_i))
			V50_AGES_B = W50_AGES / (2.0*math.sin(AA_i))
			# b) Star-dominated galaxies
			if Mstar_AGES_B > Mgas_AGES:
				V20_AGES_B = W20_AGES / (2.0*1.1*math.sin(AA_i))
				V50_AGES_B = W50_AGES / (2.0*1.1*math.sin(AA_i))
								
			# Correct for turbulence : NOT APPLIED. This gives a large scatter and causes a lot of errors 
			# for low line width objects - I recommend leaving this disabled.
			#V20_AGES_B = V20_AGES_B - 6.5
			#V50_AGES_B = V50_AGES_B - 6.5
			
			# Finally, we've got the baryonic mass !
			Mbar_AGES_B = Mstar_AGES_B + Mgas_AGES
			
			# Write all out painstakingly-obtained corrected values to a file
			ComparisonFile.write(ID+' '+str(V20_AGES_B)+' '+str(Mgas_AGES)+' '+str(Mstar_AGES_B)+' '+str(Mbar_AGES_B)+' '+str(AA_Distance)+'\n')
					
			# Set the array values we'll use for plotting
			mixedagesaa_mcg_b.append(math.log10(Mbar_AGES_B))
			mixedagesaa_v20_mcg_b.append(math.log10(V20_AGES_B))
			
			# FOR TESTING - PRINT OUTLIERS TO THE TERMINAL
			# Expected Mbar at this circular velocity 
			#Mbar_exp = math.log10(47.0*V20_AGES_B**4.0)
			#if abs(Mbar_exp - math.log10(Mbar_AGES_B)) > (3.0*0.24):
			#	print(ID, math.log10(V20_AGES_B), math.log10(Mbar_AGES_B))
				
			
VirgoFile.close()
ComparisonFile.close()



# McGaugh relation Mb = 47*v_c^4. Save the extreme points to arrays for simple plotting
mcvw = []
mcbr = []
# Upper and lower values - McGaugh's estimate of 1 sigma 0.24 dex scatter
mcbr_low = []
mcbr_up  = []
# Upper and lower values for 2 sigma 0.48 dex scatter
mcbr_olow = []
mcbr_oup  = []
for vc in [10.0,320.0]:
	logvc = math.log10(vc)
	mcvw.append(logvc)
	# MCGAUGH
	mcbr.append(math.log10(47.0*vc**4.0))
	
	mcbr_low.append(math.log10(47.0*vc**4.0) - 0.24)
	mcbr_up.append(math.log10(47.0*vc**4.0) + 0.24)
	
	mcbr_olow.append(math.log10(47.0*vc**4.0) - 0.48)
	mcbr_oup.append(math.log10(47.0*vc**4.0) + 0.48)
	


# McGaugh's own sample (resolved rotation curves, all corrections done within the file and none applied here)
McGaughGalaxiesFile = open(mggalaxiesfilename,'r')
mggalxsmbar = []
mggalxsvrot = []

for line in McGaughGalaxiesFile:
	if '#' in line:
		continue
	else:
		Name, Dist, Vcirc, ErrVc, LogMstar, ErrMstar, LogMgas, ErrMgas = line.split()
		
		Mbar = (10.0**float(LogMstar)) + (10.0**float(LogMgas))
		logMbar = math.log10(Mbar)
		logVcirc = math.log10(float(Vcirc))		
		
		mggalxsvrot.append(logVcirc)
		mggalxsmbar.append(logMbar)

McGaughGalaxiesFile.close()	



# MAKE THE PLOT
# COMBINATION OF AGES AND ALFALFA DATA WHERE AVAILABLE. MCGAUGH VELOCITY AND GAS MASS CORRECTIONS, STELLAR MASSES METHOD B
# AGES for the MHI, original line width, apparent magnitudes and galactic extinction.
# Photometric and velocity corrections use the inclination from ALFALFA.
fig=pyplot.figure()
fig.set_size_inches(800.0/100.0,800.0/100.0)

pyplot.xlim((vmin,vmax))
pyplot.ylim((bmin,bmax))
pyplot.xlabel('log(V$_{circ,20}$) / km s$^{-1}$', fontsize=titlessize)
pyplot.ylabel('log(M$_{baryons}$) / M$_{\odot}$', fontsize=titlessize)

		
plot = fig.add_subplot(111)
plot.tick_params(axis='both', which='major', labelsize=numbersize)
			
ax = plt.axes()
ax.xaxis.set_minor_locator(XminorLocator)
ax.yaxis.set_minor_locator(YminorLocator)
			
ax.tick_params(bottom=True, left=True, top=True, right=True, which='major', length=10)
ax.tick_params(bottom=True, left=True, top=True, right=True, which='minor', length=5)


# McGaugh line
pyplot.plot(mcvw,mcbr, color='black', linewidth=0.5)

# McGaugh scatter lines, +/- 2sigma from McGaugh's own estimate
pyplot.plot(mcvw,mcbr_low, color='black',linestyle='--', dashes=(10, 10), linewidth=0.5)
pyplot.plot(mcvw,mcbr_up, color='black', linestyle='--', dashes=(10, 10), linewidth=0.5)
pyplot.plot(mcvw,mcbr_olow, color='black',linestyle='dotted', dashes=(1, 7), linewidth=0.5)
pyplot.plot(mcvw,mcbr_oup, color='black', linestyle='dotted', dashes=(1, 7), linewidth=0.5)

# Galaxies have both line width corrections (*1/1.1 if M* > Mg) and baryonic mass corrections (different helium
# factor in McGaugh than everyone else)
plt.scatter(mixedagesaa_v20_mcg_b, mixedagesaa_mcg_b, facecolors='black', edgecolors='black', s=50)

# McGaugh's galaxies with resolved rotation curves
plt.scatter(mggalxsvrot, mggalxsmbar, facecolors='none', edgecolors='black', s=50, linewidth=0.5)

pyplot.savefig('FinalComparisonPlot.png',dpi=100,facecolor='white',bbox_inches='tight')

pyplot.close('all')



